/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Workfile: ConnectorConfig.cpp $
   Last Modification:
    $Author: Robert $
    $Modtime: $
    $Revision: 7135 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: no
     WinCE        : no
 
   Description:
    Implementation of the connector configuration functions
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     3         20.07.15    RM       CreateRegistryKey extended by string parameter
     2         26.11.09    RM       review
     1         28.10.09    SS       created
 
**************************************************************************************/

/*****************************************************************************/
/*! \file ConnectorConfig.cpp
*   Implementation of the connector configuration functions                  */
/*****************************************************************************/

#include "stdafx.h"
#include "netXConnectorErrors.h"
#include "ConnectorConfig.h"
#include "ConnectorAPI.h"

/*****************************************************************************/
/*! \addtogroup netX_CONNECTOR_CFG netX Connector configuration              */
/*! \{                                                                       */
/*****************************************************************************/

/////////////////////////////////////////////////////////////////////////////
/// Constructor
/////////////////////////////////////////////////////////////////////////////
CConnectorConfig::CConnectorConfig( void) 
: m_cmIntfKeyCtrlMap()
, m_cmLayerKeyCtrlMap()
, m_cmIntfConfigMap()
, m_cmLayerConfigMap()
, m_csConfigNameToken("")
{
}

/////////////////////////////////////////////////////////////////////////////
/// Constructor
/////////////////////////////////////////////////////////////////////////////
CConnectorConfig::CConnectorConfig( const UUID* ptUUID, const char* szIntfNameToken) 
: m_cmIntfKeyCtrlMap()
, m_cmLayerKeyCtrlMap()
, m_cmIntfConfigMap()
, m_cmLayerConfigMap()
, m_csConfigNameToken("")
{

  /*  Register configuration keys, valid for every connector */
  
  /*                 Key scope    Name of config key             Default   Min   Max    */
  /*                 ----------   ----------------------------   -------   ---   -----  */
  RegisterConfigKey( eLAYER     , NXCON_COMMON_ENABLED         , 1       , 0   , 1     );
  RegisterConfigKey( eINTERFACE , NXCON_COMMON_SENDTIMEOUT     , 1000    , 100 , 60000 );
  RegisterConfigKey( eINTERFACE , NXCON_COMMON_RESETTIMEOUT    , 2000    , 100 , 60000 );
  RegisterConfigKey( eINTERFACE , NXCON_COMMON_KEEPALIVETIMEOUT, 2000    , 100 , 60000 );
  RegisterConfigKey( eINTERFACE , NXCON_COMMON_EXCLUDE         , 0       , 0   , 1     );

  m_csConfigNameToken = szIntfNameToken;

  m_tUUID             = *ptUUID;
  
  InitializeCriticalSection(&m_tcsConfigLock);
}

/////////////////////////////////////////////////////////////////////////////
/// Copy Constructor
/////////////////////////////////////////////////////////////////////////////
CConnectorConfig::CConnectorConfig ( const CConnectorConfig& cConnectorConfig)
{
  *this = cConnectorConfig;
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CConnectorConfig::~CConnectorConfig( void)
{
  m_cmIntfKeyCtrlMap.clear();

  m_cmLayerKeyCtrlMap.clear();
  
  DeleteCriticalSection(&m_tcsConfigLock);
}

/////////////////////////////////////////////////////////////////////////////
/// Assignment Operator
/////////////////////////////////////////////////////////////////////////////
CConnectorConfig& CConnectorConfig::operator=( const CConnectorConfig& cSrc)
{
  /* check for assignment to self */
  if(this == &cSrc) 
    return *this; 

  m_cmIntfConfigMap   = cSrc.m_cmIntfConfigMap;
  m_cmIntfKeyCtrlMap  = cSrc.m_cmIntfKeyCtrlMap;
  m_cmLayerConfigMap  = cSrc.m_cmLayerConfigMap;
  m_cmLayerKeyCtrlMap = cSrc.m_cmLayerKeyCtrlMap;
  m_csConfigNameToken = cSrc.m_csConfigNameToken;
  m_tUUID             = cSrc.m_tUUID;
  
  InitializeCriticalSection(&m_tcsConfigLock);

  return *this;
}

/////////////////////////////////////////////////////////////////////////////
/// Request connector enable/disable
///   \return true if connector enabled
/////////////////////////////////////////////////////////////////////////////
bool  CConnectorConfig::IsConnectorEnabled( void)
{
  bool  fRet                 = true;
  CONFIGKEY_MAP cmKeyMap;

  if (NXCON_NO_ERROR == LoadLayerConfig( cmKeyMap))
  {
    fRet = atoi(cmKeyMap.find(NXCON_COMMON_ENABLED)->second)?true:false;
  }

  return fRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Register interface configuration key
///   \param eKeyScope   eLAYER or eINTERFACE
///   \param szKey       Name of key to register
///   \param dwDefault   Default value
///   \param dwMin       Default min value
///   \param dwMax       Default max value
///   \return NXCON_CONF_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::RegisterConfigKey ( NX_CONF_KEY_SCOPE_E eKeyScope, const char* szKey, DWORD dwDefault, DWORD dwMin, DWORD dwMax)
{
  long        lRet           = NXCON_NO_ERROR;
  KEYCTRL_T   tKeyCtrl       = {};
  KEYCTRL_MAP *pcmKeyCtrlMap = NULL;

  if (!szKey)
    return NXCON_DRV_INVALID_PARAMETER;

  /* Select interface or layer config map */
  switch (eKeyScope)
  {
    case eINTERFACE:
      pcmKeyCtrlMap = &m_cmIntfKeyCtrlMap;
      break;
    case eLAYER:
      pcmKeyCtrlMap = &m_cmLayerKeyCtrlMap;
      break;
    default:
      return NXCON_DRV_INVALID_PARAMETER;
      break;
  }

  /* Assign default value for config key */
  tKeyCtrl.csDefault.Format(_T("%d"), dwDefault);

  /* Assign maximum and minimum value for config key */
  tKeyCtrl.dwMin  = dwMin;
  tKeyCtrl.dwMax  = dwMax;

  if (pcmKeyCtrlMap->end() == pcmKeyCtrlMap->find(szKey))
    pcmKeyCtrlMap->insert(std::make_pair(szKey, tKeyCtrl));
  else
    (*pcmKeyCtrlMap)[szKey] = tKeyCtrl;

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Register interface configuration key
///   \param eKeyScope    eLAYER or eINTERFACE
///   \param szKey        Name of key to register
///   \param szDefault    Default value
///   \param szRegExp     Regular expression to verify value
///   \return NXCON_CONF_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::RegisterConfigKey ( NX_CONF_KEY_SCOPE_E eKeyScope, const char* szKey, const char* szDefault, const char* szRegExp)
{
  long        lRet           = NXCON_NO_ERROR;
  KEYCTRL_T   tKeyCtrl       = {};
  KEYCTRL_MAP *pcmKeyCtrlMap = NULL;

  if (!szKey)
    return NXCON_DRV_INVALID_PARAMETER;

  /* select interface or layer config map */
  switch (eKeyScope)
  {
    case eINTERFACE:
      pcmKeyCtrlMap = &m_cmIntfKeyCtrlMap;
      break;
    case eLAYER:
      pcmKeyCtrlMap = &m_cmLayerKeyCtrlMap;
      break;
    default:
      return NXCON_DRV_INVALID_PARAMETER;
      break;
  }

  /* Assign default value for config key */
  tKeyCtrl.csDefault   = szDefault;

  /* Assign regular expression to verify config key value */
  if (szRegExp)
    tKeyCtrl.csRegExp  = szRegExp;

  if (pcmKeyCtrlMap->end() == pcmKeyCtrlMap->find(szKey))
    pcmKeyCtrlMap->insert(std::make_pair(szKey, tKeyCtrl));
  else
    (*pcmKeyCtrlMap)[szKey] = tKeyCtrl;

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Get maximum key value
///   \param eKeyScope eLAYER or eINTERFACE
///   \param szKey     Key name
///   \return Maxmimum value for key 
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::GetMaxVal ( NX_CONF_KEY_SCOPE_E eKeyScope, const char* szKey)
{
  KEYCTRL_MAP *pcmKeyCtrlMap = NULL;

  /* Select interface or layer config map */
  switch (eKeyScope)
  {
    case eINTERFACE:
      pcmKeyCtrlMap = &m_cmIntfKeyCtrlMap;
      break;
    case eLAYER:
      pcmKeyCtrlMap = &m_cmLayerKeyCtrlMap;
      break;
    default:
      return 0;
      break;
  }
  
  /* Find key in config map */
  KEYCTRL_MAP::iterator iterKeyCtrl = pcmKeyCtrlMap->find(szKey);

  /* Return maximum value for config key */
  if (pcmKeyCtrlMap->end() != iterKeyCtrl)
    return iterKeyCtrl->second.dwMax;

  return 0;
}

/////////////////////////////////////////////////////////////////////////////
/// Get minimum key value
///   \param eKeyScope eLAYER or eINTERFACE
///   \param szKey     Key name
///   \return Minimum value for key 
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::GetMinVal ( NX_CONF_KEY_SCOPE_E eKeyScope, const char* szKey)
{
  KEYCTRL_MAP *pcmKeyCtrlMap = NULL;

  /* select interface or layer config map */
  switch (eKeyScope)
  {
    case eINTERFACE:
      pcmKeyCtrlMap = &m_cmIntfKeyCtrlMap;
      break;
    case eLAYER:
      pcmKeyCtrlMap = &m_cmLayerKeyCtrlMap;
      break;
    default:
      return 0;
      break;
  }
  
  /* Find key in config map */
  KEYCTRL_MAP::iterator iterKeyCtrl = pcmKeyCtrlMap->find(szKey);
  
  /* Return minimum value for config key */
  if (pcmKeyCtrlMap->end() != iterKeyCtrl)
    return iterKeyCtrl->second.dwMin;
  else
   return 0;
}

/////////////////////////////////////////////////////////////////////////////
/// Store layer configuration
///   \param eKeyScope eLAYER or eINTERFACE
///   \param cmKeyMap  Map containing default configuration keys
///   \return NXCON_CONF_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::GetDefaultConfig ( NX_CONF_KEY_SCOPE_E eKeyScope, CONFIGKEY_MAP& cmKeyMap)
{
  long        lRet           = NXCON_NO_ERROR;
  KEYCTRL_MAP *pcmKeyCtrlMap = NULL;

  /* Select interface or layer config map */
  switch (eKeyScope)
  {
    case eINTERFACE:
      pcmKeyCtrlMap = &m_cmIntfKeyCtrlMap;
      break;
    case eLAYER:
      pcmKeyCtrlMap = &m_cmLayerKeyCtrlMap;
      break;
    default:
      return NXCON_DRV_INVALID_PARAMETER;
      break;
  }  

  cmKeyMap.clear();

  for (KEYCTRL_MAP::iterator iterKeyCtrl = pcmKeyCtrlMap->begin(); iterKeyCtrl != pcmKeyCtrlMap->end(); ++iterKeyCtrl) 
    cmKeyMap.insert(std::make_pair(iterKeyCtrl->first, iterKeyCtrl->second.csDefault));

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Create interface name from connector identifier (e.g. TCP0)
///   \return Interface name
/////////////////////////////////////////////////////////////////////////////
CString CConnectorConfig::CreateInterfaceName ( void)
{
  CString csIntfName;
  int     iIntfIdx = 0;

  do
  {
    csIntfName.Format("%s%d", m_csConfigNameToken, iIntfIdx++);

  } while ((iIntfIdx < NXCON_MAX_CONFIGURATION_CNT) && (m_cmIntfConfigMap.end() != m_cmIntfConfigMap.find(csIntfName)));
  
  return csIntfName;
}

/////////////////////////////////////////////////////////////////////////////
/// Check interface name
///   \param  szInterfaceName Interface name to check
///   \return TRUE if interface name matches the pattern (e.g. COM0)
/////////////////////////////////////////////////////////////////////////////
BOOL CConnectorConfig::CheckInterfaceName ( const char* szInterfaceName)
{
  BOOL        fRet           = FALSE;
  std::string sInterfaceName = szInterfaceName;

  /* Check if registry key belongs to our configuration.
      The interface name must match the pattern <m_csConfigNameToken><Index>
      (e.g. IPRANGE5). */
  if (0 == sInterfaceName.compare(0, m_csConfigNameToken.GetLength(), m_csConfigNameToken))
  {
    /* Check if config name token is followed only by digits */
    std::string szIdx = sInterfaceName.substr(m_csConfigNameToken.GetLength(), std::string::npos - m_csConfigNameToken.GetLength());
    if ( (!szIdx.empty())                                         &&
         (string::npos == szIdx.find_first_not_of("0123456789"))  &&
         (atoi(szIdx.c_str()) < NXCON_MAX_CONFIGURATION_CNT)      )
    {
      fRet = TRUE;
    }
  }

  return fRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Create interface configuration
///   \param szName Name of interface
///   \return NXCON_CONF_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::CreateInterfaceConfig ( const char* szName)
{
  long          lRet = NXCON_NO_ERROR;
  CONFIGKEY_MAP cmKeyMap;

  /* Create default configuration for the new interface */
  if( NXCON_NO_ERROR == (lRet = GetDefaultConfig(eINTERFACE, cmKeyMap)))
  {
    /* Store default interface config map */
    lRet = StoreInterfaceConfig(szName, cmKeyMap);
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Delete interface configuration from interface map
///   \param szName  Interface name
///   \return NXCON_CONF_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::RemoveInterfaceConfig ( const char* szName)
{
  long  lRet = NXCON_NO_ERROR;
  
  if (!szName)
    return NXCON_DRV_INVALID_PARAMETER;

  /* Delete keys associated with dataset from interface map */
  INTFCONFIG_MAP::iterator iterIntf = m_cmIntfConfigMap.find(szName);
  if (m_cmIntfConfigMap.end() != iterIntf)
  {
    m_cmIntfConfigMap.erase(iterIntf);
  } else
  {
    lRet = NXCON_CONF_INVALID_INTERFACE;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Load interface configuration
///   \param szName   Name of interface
///   \param cmKeyMap Map to store interface configuration keys
///   \return NXCON_CONF_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::LoadInterfaceConfig ( const char* szName, CONFIGKEY_MAP& cmKeyMap)
{
  long    lRet              = NXCON_NO_ERROR;

  if ((!szName) || (0 == strlen(szName)))
    return NXCON_DRV_INVALID_PARAMETER;

  if (m_cmIntfConfigMap.end() != m_cmIntfConfigMap.find(szName))
  {
    cmKeyMap = m_cmIntfConfigMap[szName];
  } else
  {
    lRet = NXCON_CONF_INVALID_INTERFACE;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Load layer configuration
///   \param cmKeyMap Map to store layer configuration keys
///   \return NXCON_CONF_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::LoadLayerConfig ( CONFIGKEY_MAP& cmKeyMap)
{
  long lRet = NXCON_NO_ERROR;

  if (m_cmLayerConfigMap.empty())
  {
    lRet = GetDefaultConfig(eLAYER, m_cmLayerConfigMap);
  }

  cmKeyMap = m_cmLayerConfigMap;

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Store interface configuration
///   \param szName   Name of interface
///   \param cmKeyMap Map containing interface configuration keys
///   \return NXCON_CONF_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::StoreInterfaceConfig ( const char* szName, CONFIGKEY_MAP& cmKeyMap)
{
  long lRet = NXCON_NO_ERROR;

  if ((!szName) || (0 == strlen(szName)))
    return NXCON_DRV_INVALID_PARAMETER;

  /* Check if keys in map are known and registered via RegisterKey() */
  for (CONFIGKEY_MAP::iterator iterKeys = cmKeyMap.begin(); iterKeys != cmKeyMap.end(); ++iterKeys) 
  {
    KEYCTRL_MAP::iterator iterKeyCtrl = m_cmIntfKeyCtrlMap.find(iterKeys->first);
    if (m_cmIntfKeyCtrlMap.end() == iterKeyCtrl)
      return NXCON_CONF_INVALID_KEY;
    
    if (!VerifyValue(iterKeys->second, &iterKeyCtrl->second))
      return NXCON_CONF_INVALID_VALUE;
  }

  /* Find interface in configuration map */
  INTFCONFIG_MAP::iterator iterIntf = m_cmIntfConfigMap.find(szName);
  if (m_cmIntfConfigMap.end() == iterIntf)
  {
    /* Configuration does not exist */
    CONFIGKEY_MAP cmDefaultKey;

    /* Missing config keys are set to default */
    if( NXCON_NO_ERROR == (lRet = GetDefaultConfig(eINTERFACE, cmDefaultKey)))
    {
      cmKeyMap.insert(cmDefaultKey.begin(), cmDefaultKey.end());

      /* Insert new config in the configuration map */
      m_cmIntfConfigMap.insert(make_pair(szName, cmKeyMap));
    }

  } else
  {
    for (CONFIGKEY_MAP::iterator iterKeys = cmKeyMap.begin(); iterKeys != cmKeyMap.end(); ++iterKeys) 
    {
      CONFIGKEY_MAP::iterator iterCurrentKey = iterIntf->second.find(iterKeys->first);
      if ( iterKeys->second != iterCurrentKey->second)
      {
        iterCurrentKey->second = iterKeys->second;
      }
    }
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Store layer configuration
///   \param cmKeyMap Map containing layer configuration keys
///   \return NXCON_CONF_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::StoreLayerConfig ( CONFIGKEY_MAP& cmKeyMap)
{
  long lRet = NXCON_NO_ERROR;

  /* Check if keys in map are known and registered via RegisterKey() */
  for (CONFIGKEY_MAP::iterator iterKeys = cmKeyMap.begin(); iterKeys != cmKeyMap.end(); ++iterKeys) 
  {
    KEYCTRL_MAP::iterator iterKeyCtrl = m_cmLayerKeyCtrlMap.find(iterKeys->first);
    if (m_cmLayerKeyCtrlMap.end() == iterKeyCtrl)
      return NXCON_CONF_INVALID_KEY;

    if (!VerifyValue(iterKeys->second, &iterKeyCtrl->second))
      return NXCON_CONF_INVALID_VALUE;
  }

  for (CONFIGKEY_MAP::iterator iterKeys = cmKeyMap.begin(); iterKeys != cmKeyMap.end(); ++iterKeys) 
  {
    CONFIGKEY_MAP::iterator iterCurrentKey = m_cmLayerConfigMap.find(iterKeys->first);
    if ( iterKeys->second != iterCurrentKey->second)
    {
      iterCurrentKey->second = iterKeys->second;
    }
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Verfify value
///   \param szValue   Key value
///   \param ptKeyCtrl Key control struct holding value constraints
///   \return TRUE if value matching the regular expression
/////////////////////////////////////////////////////////////////////////////
BOOL CConnectorConfig::VerifyValue ( const char* szValue, PKEYCTRL_T ptKeyCtrl)
{
  BOOL fRet = TRUE;

  if (!szValue)
  {
    fRet = FALSE;

  } else
  {
    if ( !ptKeyCtrl->csRegExp.IsEmpty())
    {
      /* Check if key value is valid */
      CAtlRegExp<>         cREDataset;                /* regular expression object */
      CAtlREMatchContext<> cRMDataset;                /* match context object */

      cREDataset.Parse( ptKeyCtrl->csRegExp);         /* set up regular expression */
      fRet = cREDataset.Match( szValue, &cRMDataset);

    } else if (ptKeyCtrl->dwMin <= ptKeyCtrl->dwMax)
    {
      DWORD dwVal = atoi( szValue);
      if ( (dwVal > ptKeyCtrl->dwMax) || (dwVal < ptKeyCtrl->dwMin))
        fRet = FALSE;
    }
  }

  return fRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Create Registry key for storing configuration
///   \param cRegKey 
///   \return ERROR_SUCCESS if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::CreateRegistryKey ( CString& szRegInfo, CRegKey& cRegKey)
{
  CString szRegKey;
  szRegKey.Format(_T("%s\\{%X-%X-%X-%X%X-%X%X%X%X%X%X}"), NXCON_REGKEY_BASE, m_tUUID.Data1,    m_tUUID.Data2, 
                                                                             m_tUUID.Data3,    m_tUUID.Data4[0], 
                                                                             m_tUUID.Data4[1], m_tUUID.Data4[2], 
                                                                             m_tUUID.Data4[3], m_tUUID.Data4[4], 
                                                                             m_tUUID.Data4[5], m_tUUID.Data4[6], 
                                                                             m_tUUID.Data4[7]);
  szRegInfo.Format( _T("HKEY_CURRENT_USER\\%s"), NXCON_REGKEY_BASE);
  return cRegKey.Create(HKEY_CURRENT_USER, szRegKey);;
}

/////////////////////////////////////////////////////////////////////////////
/// Load interface and layer configuration from windows registry
///   \return NXCON_CONF_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::LoadFromRegistry (CString& csRegPath)
{
  long          lRet                 = NXCON_NO_ERROR;
  char          szIntfName[MAX_PATH] = "";
  unsigned long ulSize               = sizeof(szIntfName)/sizeof(*szIntfName);
  int           iIdx                 = 0;
  CRegKey       cRegBaseKey;

  if (ERROR_SUCCESS != (lRet = CreateRegistryKey(csRegPath, cRegBaseKey)))
  {
    lRet = NXCON_CONF_READ_FAILED;
  } else
  {
    /* Load layer configuration from registry */
    m_cmLayerConfigMap.clear();
    for (KEYCTRL_MAP::iterator iterKeyCtrl = m_cmLayerKeyCtrlMap.begin(); iterKeyCtrl != m_cmLayerKeyCtrlMap.end(); ++iterKeyCtrl) 
    {
      char szValue[MAX_PATH] = "";
      unsigned long ulSize   = sizeof(szValue)/sizeof(*szValue);
      
      /* Obtain value from registry */
      if ( ( ERROR_SUCCESS != cRegBaseKey.QueryStringValue( iterKeyCtrl->first, szValue, &ulSize)) ||
           ( 0 == strlen(szValue))                                                                 ) 
      {
        /* Registry key does not exist or is empty -> use default value */
        m_cmLayerConfigMap.insert(std::make_pair(iterKeyCtrl->first, iterKeyCtrl->second.csDefault));

        /* Store default value in registry */
        cRegBaseKey.SetStringValue( iterKeyCtrl->first, iterKeyCtrl->second.csDefault, REG_SZ);

      } else
      {
        if (VerifyValue(szValue, &iterKeyCtrl->second))
          m_cmLayerConfigMap.insert(std::make_pair(iterKeyCtrl->first, szValue));
        else
          m_cmLayerConfigMap.insert(std::make_pair(iterKeyCtrl->first, iterKeyCtrl->second.csDefault));
      }
    }

    /* Load interface configuration from registry */
    m_cmIntfConfigMap.clear();

    while ( ERROR_SUCCESS == cRegBaseKey.EnumKey(iIdx++, szIntfName, &ulSize))
    {
      if ( !CheckInterfaceName(szIntfName))
      {
        /* Signal user that something is wrong with the configuration in the registry */
        lRet = NXCON_CONF_READ_FAILED;
      } else
      {
        CRegKey cRegDevKey;
        if ( ERROR_SUCCESS == cRegDevKey.Open(HKEY(cRegBaseKey), szIntfName))
        {
          CONFIGKEY_MAP cmKeyMap;

          /* Iterate over valid configuration keys */
          for (KEYCTRL_MAP::iterator iterKeyCtrl = m_cmIntfKeyCtrlMap.begin(); iterKeyCtrl != m_cmIntfKeyCtrlMap.end(); ++iterKeyCtrl) 
          {
            char szValue[MAX_PATH] = "";
            unsigned long ulSize   = sizeof(szValue)/sizeof(*szValue);
            
            /* Obtain value from registry */
            if ( ( ERROR_SUCCESS != cRegDevKey.QueryStringValue( iterKeyCtrl->first, szValue, &ulSize)) ||
                 ( 0 == strlen(szValue))                                                                )
            {
              /* Registry key does not exist or is empty -> use default value */
              cmKeyMap.insert(std::make_pair(iterKeyCtrl->first, iterKeyCtrl->second.csDefault));

              /* Store default value in registry */
              cRegDevKey.SetStringValue( iterKeyCtrl->first, iterKeyCtrl->second.csDefault, REG_SZ);

            } else
            {
              if (VerifyValue(szValue, &iterKeyCtrl->second))
                cmKeyMap.insert(std::make_pair(iterKeyCtrl->first, szValue));
              else
                cmKeyMap.insert(std::make_pair(iterKeyCtrl->first, iterKeyCtrl->second.csDefault));
            }
          }
          
          /* Store interface in map */
          m_cmIntfConfigMap.insert(make_pair(szIntfName, cmKeyMap));

          cRegDevKey.Close();
        }
      }

      ulSize = sizeof(szIntfName)/sizeof(*szIntfName);
    }

    /* Close registry base key */
    cRegBaseKey.Close();
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Store interface and layer configuration to windows registry
///   \return NXCON_CONF_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::StoreToRegistry (CString& csRegPath)
{
  long          lRet                 = NXCON_NO_ERROR;
  char          szIntfName[MAX_PATH] = "";
  unsigned long ulSize               = sizeof(szIntfName)/sizeof(*szIntfName);
  CRegKey       cRegBaseKey;

  if (ERROR_SUCCESS != CreateRegistryKey(csRegPath, cRegBaseKey))
  {
    lRet = NXCON_CONF_WRITE_FAILED;
  } else
  {
    /* Remove interface configuration from windows registry... */
    while ( ERROR_SUCCESS == cRegBaseKey.EnumKey(0, szIntfName, &ulSize))
    {
      if(ERROR_SUCCESS != cRegBaseKey.DeleteSubKey(szIntfName))
      {
        /* Error accessing registry */
        lRet = NXCON_CONF_WRITE_FAILED;
        break;
      }
      ulSize = sizeof(szIntfName) / sizeof(*szIntfName);
    }

    /* ... and write new interface configuration to windows registry */
    for (INTFCONFIG_MAP::iterator iterIntf = m_cmIntfConfigMap.begin(); iterIntf != m_cmIntfConfigMap.end(); ++iterIntf) 
    {
      CRegKey  cRegDevKey;
      if ( ERROR_SUCCESS != cRegDevKey.Create(HKEY(cRegBaseKey), iterIntf->first))
      {
        lRet = NXCON_CONF_WRITE_FAILED;
        break;
      } else
      {
        for (CONFIGKEY_MAP::iterator iterKeys = iterIntf->second.begin(); iterKeys != iterIntf->second.end(); ++iterKeys) 
        {
          if ( ERROR_SUCCESS != cRegDevKey.SetStringValue( iterKeys->first, iterKeys->second, REG_SZ))
          {
            lRet = NXCON_CONF_WRITE_FAILED;
            break;
          }
        }
        cRegDevKey.Close();
      }
    }

    /* Store layer configuration to windows registry */
    if ( NXCON_NO_ERROR == lRet)
    {
      for (CONFIGKEY_MAP::iterator iterKeys = m_cmLayerConfigMap.begin(); iterKeys != m_cmLayerConfigMap.end(); ++iterKeys) 
      {
        if ( ERROR_SUCCESS != cRegBaseKey.SetStringValue( iterKeys->first, iterKeys->second, REG_SZ))
        {
          lRet = NXCON_CONF_WRITE_FAILED;
          break;
        }
      }
    }

    cRegBaseKey.Close();
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Parse configuration string (e.g. "DEVNAME=IP0,IPADDR=192.168.1.5")
///   \param szConfig     Configuration string
///   \return NXAPI_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::ParseString ( const char* szConfig)
{ 
  long lRet = NXCON_NO_ERROR;

  if (!szConfig)
    return NXCON_DRV_INVALID_PARAMETER;

  CConfigString cConfigString( szConfig);

  /* Backup current configuration */
  INTFCONFIG_MAP cmIntfConfigBackup = m_cmIntfConfigMap;
  CONFIGKEY_MAP cmLayerConfigBackup = m_cmLayerConfigMap;

  /* Clear Interface Configuration */
  m_cmIntfConfigMap.clear();
  m_cmLayerConfigMap.clear();
  
  /* Set default Layer Configuration */
  if( NXCON_NO_ERROR == (lRet = GetDefaultConfig(eLAYER, m_cmLayerConfigMap)))
  { 
    /* Get new Configuration from config string */
    for(int iDatasetIdx = 0; iDatasetIdx < cConfigString.GetDatasetCnt(); iDatasetIdx++)
    {
      CONFIGKEY_MAP cmKeyValMap;

      if (NXCON_NO_ERROR == (lRet = cConfigString.Parse( iDatasetIdx, cmKeyValMap)))
      {
        CONFIGKEY_MAP::iterator iterName;

        if ( cmKeyValMap.end() == (iterName = cmKeyValMap.find(NXCON_CONFIGKEY_DEVNAME)))
        {
          lRet = NXCON_CONF_INVALID_KEY;
          break;

        } else
        {
          CString csName = iterName->second;
          cmKeyValMap.erase(iterName);
          /* If interface name is "GLOBAL", this is our layer configuration */
          if (0 == csName.CompareNoCase(NXCON_DEVNAME_LAYER))
          {
            if (NXCON_NO_ERROR != (lRet = StoreLayerConfig( cmKeyValMap)))
              break;

          /* This is an interface configuration if the interface name 
            matches the pattern <m_csConfigNameToken><Number> */
          } else if (CheckInterfaceName(csName))
          {
            if (NXCON_NO_ERROR != (lRet = StoreInterfaceConfig(csName, cmKeyValMap)))
              break;

          /* Otherwise the supplied name is invalid */
          } else
          {
            lRet = NXCON_CONF_INVALID_VALUE;
            break;
          }
        }
      }
    } 
  }

  /* Restore original configuration */ 
  if (NXCON_NO_ERROR != lRet)
  {
    /* Configuration string was erroneous -> restore backup */
    m_cmIntfConfigMap  = cmIntfConfigBackup;
    m_cmLayerConfigMap = cmLayerConfigBackup;
  }
  
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Create configuration string
///   \param csConfigString String buffer
///   \return NXCON_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConnectorConfig::CreateString( CString& csConfigString)
{
  long          lRet = NXCON_NO_ERROR;
  CONFIGKEY_MAP cmKeyMap;

  if ( NXCON_NO_ERROR != (lRet = LoadLayerConfig( cmKeyMap)))
    /* We cant get a map, just return an empty string */
    return lRet;

  /* Write layer configuration to string */
  /* Layer configuration is stored in a single dataset with interface name GLOBAL
     (DEVNAME=GLOBAL) */
  csConfigString.AppendFormat(_T("%s=%s"), NXCON_CONFIGKEY_DEVNAME, NXCON_DEVNAME_LAYER);

  for (CONFIGKEY_MAP::iterator iterKeys = cmKeyMap.begin(); iterKeys != cmKeyMap.end(); ++iterKeys) 
  {
    /* Append key-value pairs (e.g. ENABLED=1) */
    csConfigString.AppendFormat(_T("%s%s=%s"), NXCON_KEYVALUE_SEPERATOR, iterKeys->first, iterKeys->second);
  }
  
  /* Seperate from interface config datasets */
  csConfigString.Append( NXCON_DATASET_SEPERATOR);

  /* Write interface configuration to string */
  for (iterator iterIntfName = begin(); iterIntfName != end(); ++iterIntfName) 
  {
    CONFIGKEY_MAP cmKeyMap;
    LoadInterfaceConfig(*iterIntfName, cmKeyMap);

    /* Append interface name (e.g. DEVNAME=COM1) */
    csConfigString.AppendFormat(_T("%s=%s"), NXCON_CONFIGKEY_DEVNAME, *iterIntfName);
    
    for (CONFIGKEY_MAP::iterator iterKeys = cmKeyMap.begin(); iterKeys != cmKeyMap.end(); ++iterKeys) 
    {
      /* Append key-value pairs (e.g. BAUDRATE=115200) */
      csConfigString.AppendFormat(_T("%s%s=%s"), NXCON_KEYVALUE_SEPERATOR, iterKeys->first, iterKeys->second);
    }

    /* Append seperator for next interface config dataset */
    csConfigString.Append( NXCON_DATASET_SEPERATOR);
  }
  
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Constructor
/////////////////////////////////////////////////////////////////////////////
CConfigString::CConfigString ( const char* szConfig)
: m_csConfigString("")
, m_iDatasetCnt(0)
{
  CAtlRegExp<>::RECHAR* pszDataset;
  CAtlRegExp<>          cREDataset;  /* Regular expression object */
  CAtlREMatchContext<>  cRMDataset;  /* Match context object */
  
  m_csConfigString = szConfig;
  pszDataset       = m_csConfigString.GetBuffer();  
  cREDataset.Parse( NXCON_REGEXP_DATASETSPLIT);   /* Set up regular expression */

  m_iDatasetCnt = 0;

  while ( (pszDataset < m_csConfigString.GetBuffer() + m_csConfigString.GetLength())                       &&
          (cREDataset.Match( pszDataset, &cRMDataset, (const CAtlREMatchContext<>::RECHAR**)&pszDataset )) )
  {
    m_iDatasetCnt++;
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CConfigString::~CConfigString ( )
{
}

/////////////////////////////////////////////////////////////////////////////
/// Parse configuration string and store keys in a map
///   \param iDatasetIdx   Dataset index
///   \param cmKeyMap Map containing keys and values 
///   \return NXCON_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CConfigString::Parse ( unsigned int iDatasetIdx, CONFIGKEY_MAP& cmKeyMap)
{
  CAtlRegExp<>::RECHAR*      pszDataset  = m_csConfigString.GetBuffer();
  unsigned int iDatasetCnt = 0;
  long         lRet        = NXCON_CONF_INVALID_VALUE;

  CAtlRegExp<>         cREDataset;  /* Regular expression object */
  CAtlREMatchContext<> cRMDataset;  /* Match context object */
  cREDataset.Parse( NXCON_REGEXP_DATASETSPLIT);   /* Set up regular expression */

  while ((pszDataset < m_csConfigString.GetBuffer() + m_csConfigString.GetLength()) &&
         (cREDataset.Match( pszDataset, &cRMDataset, (const CAtlREMatchContext<>::RECHAR**)&pszDataset)) )
  {
    if ( iDatasetCnt == iDatasetIdx)
    {
      const CAtlRegExp<>::RECHAR* pszValuepair    = 0;
      const CAtlRegExp<>::RECHAR* pszValuepairEnd = 0;
      CAtlREMatchContext<> cRMValuepair;       /* Match context object */
      CAtlRegExp<>         cREValuepair;       /* regular expression object */
    
      cRMDataset.GetMatch(0, &pszValuepair, &pszValuepairEnd);

      cREValuepair.Parse( NXCON_REGEXP_KEYVALUESPLIT); // set up regular expression
      while ( (pszValuepair<pszValuepairEnd) && (cREValuepair.Match( pszValuepair, &cRMValuepair, (const CAtlREMatchContext<>::RECHAR**)&pszValuepair )))
      {
        const CAtlRegExp<>::RECHAR* szStart = 0;
        const CAtlRegExp<>::RECHAR* szEnd   = 0;
        CString         csConfigKey;
        cRMValuepair.GetMatch(0, &szStart, &szEnd);
        csConfigKey.SetString(szStart, (int)(szEnd - szStart));
        cRMValuepair.GetMatch(1, &szStart, &szEnd);
        cmKeyMap[csConfigKey].SetString(szStart, (int)(szEnd - szStart));
      }
      
      lRet = NXCON_NO_ERROR;      
      break;
    }
    
    ++iDatasetCnt;
  }

  return lRet;
}

/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/
